const db = require("../config/db");
const bcrypt = require("bcryptjs");
const jwt = require("jsonwebtoken");
const fs = require("fs");
const path = require("path");
const { JWT_SECRET } = require("../middleware/authMiddleware");
const logger = require("../utils/logger");

/**
 * Admin Login
 * POST /api/admin/login
 */
exports.adminLogin = async (req, res) => {
  try {
    const { username, password } = req.body;

    if (!username || !password) {
      return res.status(400).json({
        error: "Validation Error",
        message: "Username and password are required"
      });
    }

    // Get admin credentials from environment variables
    const ADMIN_USERNAME = (process.env.ADMIN_USERNAME || "admin").trim();
    const ADMIN_PASSWORD = (process.env.ADMIN_PASSWORD || "admin123").trim();
    const isProduction = process.env.NODE_ENV === 'production';
    
    // Trim input values
    const inputUsername = (username || "").trim();
    const inputPassword = (password || "").trim();

    // Debug logging (only in development)
    if (!isProduction) {
      logger.debug(`Login attempt - Username match: ${inputUsername === ADMIN_USERNAME}, Expected: ${ADMIN_USERNAME}, Got: ${inputUsername}`);
      logger.debug(`Password type: ${ADMIN_PASSWORD.startsWith('$2') ? 'Hashed' : 'Plain text'}`);
    }

    // Verify credentials
    if (inputUsername !== ADMIN_USERNAME) {
      logger.warn(`Login failed - Username mismatch. Expected: "${ADMIN_USERNAME}", Got: "${inputUsername}"`);
      return res.status(401).json({
        error: "Unauthorized",
        message: "Invalid username or password"
      });
    }

    // Password verification
    let passwordMatch = false;
    const isPasswordHashed = ADMIN_PASSWORD && (ADMIN_PASSWORD.startsWith('$2a$') || ADMIN_PASSWORD.startsWith('$2b$') || ADMIN_PASSWORD.startsWith('$2y$'));
    
    if (isProduction) {
      // Production: Only accept hashed passwords
      if (!isPasswordHashed) {
        logger.error("ADMIN_PASSWORD in production must be a bcrypt hash");
        return res.status(500).json({
          error: "Server Error",
          message: "Server configuration error"
        });
      }
      
      try {
        passwordMatch = await bcrypt.compare(inputPassword, ADMIN_PASSWORD);
      } catch (err) {
        logger.error("Password comparison error:", err);
        return res.status(401).json({
          error: "Unauthorized",
          message: "Invalid username or password"
        });
      }
    } else {
      // Development: Accept both hashed and plain text
      if (isPasswordHashed) {
        // Password is hashed, use bcrypt
        try {
          passwordMatch = await bcrypt.compare(inputPassword, ADMIN_PASSWORD);
        } catch (err) {
          logger.error("Password comparison error:", err);
          // Fallback to plain text if bcrypt fails
          passwordMatch = inputPassword === ADMIN_PASSWORD;
        }
      } else {
        // Password is plain text, compare directly
        passwordMatch = inputPassword === ADMIN_PASSWORD;
        if (!isProduction) {
          logger.debug(`Plain text comparison - Expected: "${ADMIN_PASSWORD}", Got: "${inputPassword}", Match: ${passwordMatch}`);
        }
      }
    }

    if (!passwordMatch) {
      logger.warn(`Login failed for username: ${inputUsername} - Password mismatch`);
      if (!isProduction) {
        logger.debug(`Password comparison failed. Expected length: ${ADMIN_PASSWORD.length}, Got length: ${inputPassword.length}`);
      }
      return res.status(401).json({
        error: "Unauthorized",
        message: "Invalid username or password"
      });
    }

    // Generate JWT token
    if (!JWT_SECRET) {
      logger.error("JWT_SECRET is not defined");
      return res.status(500).json({
        error: "Server Error",
        message: "Server configuration error"
      });
    }

    const token = jwt.sign(
      { username: inputUsername, role: "admin" },
      JWT_SECRET,
      { expiresIn: process.env.JWT_EXPIRES_IN || "24h" }
    );

    logger.info(`Successful login for username: ${inputUsername}`);
    res.json({
      token,
      username: inputUsername
    });
  } catch (error) {
    logger.error("Admin login error:", error);
    
    // Make sure response hasn't been sent
    if (!res.headersSent) {
      res.status(500).json({
        error: "Server Error",
        message: "Failed to process login request"
      });
    }
  }
};

/**
 * Get All Users with Filters
 * GET /api/admin/users?search=&status=&dateFrom=&dateTo=
 */
exports.getAllUsers = async (req, res) => {
  try {
    const { search, status, lead_type, employee_id, dateFrom, dateTo } = req.query;

    // Build query with filters
    let query = `
      SELECT 
        u.id as user_id,
        u.fullname,
        u.contact,
        u.address,
        u.office_address,
        u.office_contact,
        u.work_experience,
        u.declaration_acceptance,
        u.aadhaar_number,
        u.pan_number,
        u.passport_number,
        COALESCE(u.status, 'pending') as status,
        u.lead_type,
        u.employee_id,
        u.created_at,
        e.name as employee_name,
        ud.aadhaar_file,
        ud.pan_file,
        ud.passport_file,
        ud.signature_file,
        ud.mou_pdf
      FROM users u
      LEFT JOIN user_documents ud ON u.id = ud.user_id
      LEFT JOIN employees e ON u.employee_id = e.employee_id
      WHERE 1=1
    `;

    const queryParams = [];

    // Search filter (searches in fullname, contact, aadhaar, pan)
    if (search) {
      query += ` AND (
        u.fullname LIKE ? OR 
        u.contact LIKE ? OR 
        u.aadhaar_number LIKE ? OR 
        u.pan_number LIKE ?
      )`;
      const searchPattern = `%${search}%`;
      queryParams.push(searchPattern, searchPattern, searchPattern, searchPattern);
    }

    // Status filter
    if (status && ['active', 'inactive', 'pending'].includes(status)) {
      query += ` AND COALESCE(u.status, 'pending') = ?`;
      queryParams.push(status);
    }

    // Lead type filter
    if (lead_type && ['chp', 'fsp'].includes(lead_type.toLowerCase())) {
      query += ` AND u.lead_type = ?`;
      queryParams.push(lead_type.toLowerCase());
    }

    // Employee ID filter
    if (employee_id && employee_id !== 'all') {
      const employeeIdNum = parseInt(employee_id);
      if (!isNaN(employeeIdNum)) {
        query += ` AND u.employee_id = ?`;
        queryParams.push(employeeIdNum);
      }
    }

    // Date range filters
    if (dateFrom) {
      query += ` AND DATE(u.created_at) >= ?`;
      queryParams.push(dateFrom);
    }

    if (dateTo) {
      query += ` AND DATE(u.created_at) <= ?`;
      queryParams.push(dateTo);
    }

    query += ` ORDER BY u.created_at DESC`;

    const [users] = await db.execute(query, queryParams);

    // Format response with documents structure
    const formattedUsers = users.map(user => ({
      user_id: user.user_id,
      fullname: user.fullname,
      contact: user.contact,
      address: user.address,
      office_address: user.office_address,
      office_contact: user.office_contact,
      work_experience: user.work_experience,
      declaration_acceptance: Boolean(user.declaration_acceptance),
      aadhaar_number: user.aadhaar_number,
      pan_number: user.pan_number,
      passport_number: user.passport_number,
      status: user.status || 'pending',
      lead_type: user.lead_type || null,
      employee_id: user.employee_id || null,
      employee_name: user.employee_name || null,
      created_at: user.created_at,
      documents: {
        aadhaar: user.aadhaar_file || null,
        pan: user.pan_file || null,
        passport: user.passport_file || null,
        signature: user.signature_file || null,
        mou: user.mou_pdf || null
      }
    }));

    res.json({ data: formattedUsers });
  } catch (error) {
    logger.error("Get all users error:", error);
    res.status(500).json({
      error: "Server Error",
      message: "Failed to fetch users"
    });
  }
};

/**
 * Get User by ID
 * GET /api/admin/users/:userId
 */
exports.getUserById = async (req, res) => {
  try {
    const { userId } = req.params;

    if (!userId || isNaN(userId)) {
      return res.status(400).json({
        error: "Validation Error",
        message: "Invalid user ID"
      });
    }

    const [users] = await db.execute(
      `SELECT 
        u.id as user_id,
        u.fullname,
        u.contact,
        u.address,
        u.office_address,
        u.office_contact,
        u.work_experience,
        u.declaration_acceptance,
        u.aadhaar_number,
        u.pan_number,
        u.passport_number,
        COALESCE(u.status, 'pending') as status,
        u.lead_type,
        u.employee_id,
        u.created_at,
        e.name as employee_name,
        ud.aadhaar_file,
        ud.pan_file,
        ud.passport_file,
        ud.signature_file,
        ud.mou_pdf
      FROM users u
      LEFT JOIN user_documents ud ON u.id = ud.user_id
      LEFT JOIN employees e ON u.employee_id = e.employee_id
      WHERE u.id = ?`,
      [userId]
    );

    if (users.length === 0) {
      return res.status(404).json({
        error: "Not Found",
        message: "User not found"
      });
    }

    const user = users[0];
    const formattedUser = {
      user_id: user.user_id,
      fullname: user.fullname,
      contact: user.contact,
      address: user.address,
      office_address: user.office_address,
      office_contact: user.office_contact,
      work_experience: user.work_experience,
      declaration_acceptance: Boolean(user.declaration_acceptance),
      aadhaar_number: user.aadhaar_number,
      pan_number: user.pan_number,
      passport_number: user.passport_number,
      status: user.status || 'pending',
      lead_type: user.lead_type || null,
      employee_id: user.employee_id || null,
      employee_name: user.employee_name || null,
      created_at: user.created_at,
      documents: {
        aadhaar: user.aadhaar_file || null,
        pan: user.pan_file || null,
        passport: user.passport_file || null,
        signature: user.signature_file || null,
        mou: user.mou_pdf || null
      }
    };

    res.json({ data: formattedUser });
  } catch (error) {
    logger.error("Get user by ID error:", error);
    res.status(500).json({
      error: "Server Error",
      message: "Failed to fetch user"
    });
  }
};

/**
 * Update User Status
 * PATCH /api/admin/users/:userId/status
 */
exports.updateUserStatus = async (req, res) => {
  try {
    const { userId } = req.params;
    const { status } = req.body;

    if (!userId || isNaN(userId)) {
      return res.status(400).json({
        error: "Validation Error",
        message: "Invalid user ID"
      });
    }

    if (!status || !['active', 'inactive', 'pending'].includes(status)) {
      return res.status(400).json({
        error: "Validation Error",
        message: "Status must be one of: active, inactive, pending"
      });
    }

    // Check if user exists
    const [users] = await db.execute(
      `SELECT id FROM users WHERE id = ?`,
      [userId]
    );

    if (users.length === 0) {
      return res.status(404).json({
        error: "Not Found",
        message: "User not found"
      });
    }

    // Update status
    await db.execute(
      `UPDATE users SET status = ? WHERE id = ?`,
      [status, userId]
    );

    res.json({
      success: true,
      message: `User status updated to ${status}`
    });
  } catch (error) {
    logger.error("Update user status error:", error);
    res.status(500).json({
      error: "Server Error",
      message: "Failed to update user status"
    });
  }
};

/**
 * Download User Document
 * GET /api/admin/users/:userId/documents/:documentType
 */
exports.downloadDocument = async (req, res) => {
  try {
    const { userId, documentType } = req.params;

    // Validate user ID
    if (!userId || isNaN(userId)) {
      return res.status(400).json({
        error: "Validation Error",
        message: "Invalid user ID"
      });
    }

    // Validate document type
    const validTypes = ['aadhaar', 'pan', 'passport', 'signature', 'mou'];
    if (!validTypes.includes(documentType)) {
      return res.status(400).json({
        error: "Invalid document type",
        message: `Document type must be one of: ${validTypes.join(', ')}`
      });
    }

    // Get user and document info from database
    const [users] = await db.execute(
      `SELECT u.id, u.fullname, ud.* 
       FROM users u
       LEFT JOIN user_documents ud ON u.id = ud.user_id
       WHERE u.id = ?`,
      [userId]
    );

    // Check if user exists
    if (users.length === 0) {
      return res.status(404).json({
        error: "User not found",
        message: `User with ID ${userId} not found`
      });
    }

    const user = users[0];
    
    // Map document type to database field name
    const documentFieldMap = {
      aadhaar: 'aadhaar_file',
      pan: 'pan_file',
      passport: 'passport_file',
      signature: 'signature_file',
      mou: 'mou_pdf'
    };

    const documentPath = user[documentFieldMap[documentType]];

    // Check if document exists in database
    if (!documentPath) {
      return res.status(404).json({
        error: "Document not found",
        message: `${documentType} document not uploaded for this user`
      });
    }

    // LOCAL DEVELOPMENT: Always convert database paths to local paths
    // Database may have production absolute paths, but we always use local relative paths
    let fullPath;
    const documentPathStr = String(documentPath).trim();
    const projectRoot = path.join(__dirname, "..");
    
    // Check if it's already a relative path starting with 'uploads'
    if (documentPathStr.startsWith('uploads')) {
      // Already relative path - join with project root
      fullPath = path.join(projectRoot, documentPathStr);
    } 
    // Check if it's an absolute path (from production database)
    else if (path.isAbsolute(documentPathStr)) {
      // Extract the relative part after 'uploads' directory
      // Example: /home/backendftvassets/public_html/uploads/users/John/aadhaar/file.pdf
      // Becomes: uploads/users/John/aadhaar/file.pdf
      const pathLower = documentPathStr.toLowerCase();
      const uploadsIndex = pathLower.indexOf('uploads');
      
      if (uploadsIndex !== -1) {
        // Extract everything after 'uploads' (including 'uploads' itself)
        const relativePath = documentPathStr.substring(uploadsIndex);
        // Join with project root
        fullPath = path.join(projectRoot, relativePath);
      } else {
        // If 'uploads' not found, try to find 'users' directory
        const usersIndex = pathLower.indexOf('users');
        if (usersIndex !== -1) {
          // Extract from 'users' onwards and prepend 'uploads'
          const usersPath = documentPathStr.substring(usersIndex);
          fullPath = path.join(projectRoot, "uploads", usersPath);
        } else {
          // Fallback: use the filename and try to find it
          // This shouldn't happen, but handle gracefully
          const fileName = path.basename(documentPathStr);
          fullPath = path.join(projectRoot, "uploads", fileName);
        }
      }
    } 
    // Relative path (not starting with 'uploads')
    else {
      // If it doesn't start with 'uploads', prepend it
      if (!documentPathStr.startsWith('uploads')) {
        fullPath = path.join(projectRoot, "uploads", documentPathStr);
      } else {
        fullPath = path.join(projectRoot, documentPathStr);
      }
    }

    // Normalize path separators for cross-platform compatibility (Windows/Unix)
    fullPath = path.normalize(fullPath);
    
    // Log for debugging (only in development)
    logger.debug(`Original DB path: ${documentPathStr}`);
    logger.debug(`Resolved local path: ${fullPath}`);

    // Check if file exists on server
    if (!fs.existsSync(fullPath)) {
      const fileDir = path.dirname(fullPath);
      const fileName = path.basename(fullPath);
      const dirExists = fs.existsSync(fileDir);
      const isProduction = process.env.NODE_ENV === 'production' || process.env.ENVIRONMENT === 'production';
      
      logger.error(`File not found: ${fullPath}`);
      logger.error(`Original path from DB: ${documentPath}`);
      logger.error(`Environment: ${isProduction ? 'PRODUCTION' : 'LOCAL'}`);
      logger.error(`Directory exists: ${dirExists}`);
      
      // Provide helpful error message
      let errorMessage = "File not found on server.";
      if (!isProduction) {
        // Local environment - file might only exist on production
        const userFolder = path.basename(path.dirname(path.dirname(fullPath)));
        if (!dirExists) {
          errorMessage = `Directory not found locally. The user folder '${userFolder}' may not exist in your local uploads directory. `;
          errorMessage += `Files for this user may only exist on the production server.`;
        } else {
          errorMessage = `File '${fileName}' not found locally for user '${userFolder}'. `;
          errorMessage += `The file may only exist on the production server. `;
          errorMessage += `Expected location: ${fullPath}`;
        }
      } else {
        if (!dirExists) {
          errorMessage += ` Directory does not exist: ${fileDir}`;
        } else {
          errorMessage += ` File does not exist: ${fullPath}`;
        }
        errorMessage += " Please check file path and permissions on production server.";
      }
      
      return res.status(404).json({
        error: "Document not found",
        message: errorMessage,
        details: {
          resolvedPath: fullPath,
          originalPath: documentPath,
          environment: isProduction ? 'production' : 'local',
          directoryExists: dirExists,
          fileName: fileName
        }
      });
    }

    // Get file extension and determine content type
    const ext = path.extname(fullPath).toLowerCase();
    const contentTypeMap = {
      '.pdf': 'application/pdf',
      '.jpg': 'image/jpeg',
      '.jpeg': 'image/jpeg',
      '.png': 'image/png',
      '.gif': 'image/gif',
      '.webp': 'image/webp'
    };

    const contentType = contentTypeMap[ext] || 'application/octet-stream';
    
    // Generate filename for download
    const originalFilename = path.basename(fullPath);
    const safeFullname = user.fullname.replace(/[^a-zA-Z0-9_]/g, '_');
    const downloadFilename = `${documentType}_${safeFullname}${ext}`;

    // Set response headers
    res.setHeader('Content-Type', contentType);
    res.setHeader('Content-Disposition', `attachment; filename="${downloadFilename}"`);
    
    // Add CORS headers if needed (for frontend access)
    res.setHeader('Access-Control-Expose-Headers', 'Content-Disposition');
    
    // Send file using sendFile (recommended method)
      res.sendFile(path.resolve(fullPath), (err) => {
      if (err) {
        logger.error("Error sending file:", err);
        if (!res.headersSent) {
          res.status(500).json({
            error: "Failed to serve document",
            message: "Error reading or sending file"
          });
        }
      }
    });
  } catch (error) {
    logger.error("Download document error:", error);
    if (!res.headersSent) {
      res.status(500).json({
        error: "Failed to serve document",
        message: error.message || "Server error while downloading document"
      });
    }
  }
};

/**
 * Get Admin Statistics
 * GET /api/admin/stats
 */
exports.getAdminStats = async (req, res) => {
  try {
    // Get all statistics in parallel
    const [totalUsers] = await db.execute(
      `SELECT COUNT(*) as count FROM users`
    );

    const [activeUsers] = await db.execute(
      `SELECT COUNT(*) as count FROM users WHERE COALESCE(status, 'pending') = 'active'`
    );

    const [inactiveUsers] = await db.execute(
      `SELECT COUNT(*) as count FROM users WHERE COALESCE(status, 'pending') = 'inactive'`
    );

    const [pendingUsers] = await db.execute(
      `SELECT COUNT(*) as count FROM users WHERE COALESCE(status, 'pending') = 'pending'`
    );

    // Recent registrations (last 7 days)
    const [recentRegistrations] = await db.execute(
      `SELECT COUNT(*) as count FROM users 
       WHERE DATE(created_at) >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)`
    );

    res.json({
      data: {
        totalUsers: totalUsers[0].count,
        activeUsers: activeUsers[0].count,
        inactiveUsers: inactiveUsers[0].count,
        pendingUsers: pendingUsers[0].count,
        recentRegistrations: recentRegistrations[0].count
      }
    });
  } catch (error) {
    logger.error("Get admin stats error:", error);
    res.status(500).json({
      error: "Server Error",
      message: "Failed to fetch statistics"
    });
  }
};

