const db = require("../config/db");
const logger = require("../utils/logger");

/**
 * Get All Employees
 * GET /api/admin/employees
 */
exports.getAllEmployees = async (req, res) => {
  try {
    const [employees] = await db.execute(
      `SELECT 
        employee_id,
        name,
        email,
        phone,
        designation,
        status,
        created_at,
        updated_at
      FROM employees 
      ORDER BY created_at DESC`
    );

    res.json({ data: employees });
  } catch (error) {
    logger.error("Get all employees error:", error);
    res.status(500).json({
      error: "Server Error",
      message: "Failed to fetch employees"
    });
  }
};

/**
 * Get Employee by ID
 * GET /api/admin/employees/:employeeId
 */
exports.getEmployeeById = async (req, res) => {
  try {
    const { employeeId } = req.params;

    if (!employeeId || isNaN(employeeId)) {
      return res.status(400).json({
        error: "Validation Error",
        message: "Invalid employee ID"
      });
    }

    const [employees] = await db.execute(
      `SELECT 
        employee_id,
        name,
        email,
        phone,
        designation,
        status,
        created_at,
        updated_at
      FROM employees 
      WHERE employee_id = ?`,
      [employeeId]
    );

    if (employees.length === 0) {
      return res.status(404).json({
        error: "Employee not found",
        message: `Employee with ID ${employeeId} not found`
      });
    }

    res.json({ data: employees[0] });
  } catch (error) {
    logger.error("Get employee by ID error:", error);
    res.status(500).json({
      error: "Server Error",
      message: "Failed to fetch employee"
    });
  }
};

/**
 * Create New Employee
 * POST /api/admin/employees
 */
exports.createEmployee = async (req, res) => {
  try {
    const { name, email, phone, designation } = req.body;

    // Validation
    if (!name || name.trim() === '') {
      return res.status(400).json({
        error: "Validation Error",
        message: "Name is required"
      });
    }

    // Optional email validation
    if (email && email.trim() !== '') {
      const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
      if (!emailRegex.test(email.trim())) {
        return res.status(400).json({
          error: "Validation Error",
          message: "Invalid email format"
        });
      }
    }

    // Insert employee
    const [result] = await db.execute(
      `INSERT INTO employees (name, email, phone, designation) 
       VALUES (?, ?, ?, ?)`,
      [
        name.trim(),
        email?.trim() || null,
        phone?.trim() || null,
        designation?.trim() || null
      ]
    );

    // Fetch created employee
    const [employees] = await db.execute(
      `SELECT 
        employee_id,
        name,
        email,
        phone,
        designation,
        status,
        created_at,
        updated_at
      FROM employees 
      WHERE employee_id = ?`,
      [result.insertId]
    );

    res.status(201).json({ data: employees[0] });
  } catch (error) {
    logger.error("Create employee error:", error);
    res.status(500).json({
      error: "Server Error",
      message: "Failed to create employee"
    });
  }
};

/**
 * Update Employee
 * PATCH /api/admin/employees/:employeeId
 */
exports.updateEmployee = async (req, res) => {
  try {
    const { employeeId } = req.params;
    const { name, email, phone, designation } = req.body;

    if (!employeeId || isNaN(employeeId)) {
      return res.status(400).json({
        error: "Validation Error",
        message: "Invalid employee ID"
      });
    }

    // Check if employee exists
    const [existing] = await db.execute(
      `SELECT employee_id FROM employees WHERE employee_id = ?`,
      [employeeId]
    );

    if (existing.length === 0) {
      return res.status(404).json({
        error: "Employee not found",
        message: `Employee with ID ${employeeId} not found`
      });
    }

    // Build update query dynamically
    const updates = [];
    const values = [];

    if (name !== undefined) {
      if (!name || name.trim() === '') {
        return res.status(400).json({
          error: "Validation Error",
          message: "Name cannot be empty"
        });
      }
      updates.push('name = ?');
      values.push(name.trim());
    }

    if (email !== undefined) {
      if (email && email.trim() !== '') {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        if (!emailRegex.test(email.trim())) {
          return res.status(400).json({
            error: "Validation Error",
            message: "Invalid email format"
          });
        }
        updates.push('email = ?');
        values.push(email.trim());
      } else {
        updates.push('email = ?');
        values.push(null);
      }
    }

    if (phone !== undefined) {
      updates.push('phone = ?');
      values.push(phone?.trim() || null);
    }

    if (designation !== undefined) {
      updates.push('designation = ?');
      values.push(designation?.trim() || null);
    }

    if (updates.length === 0) {
      return res.status(400).json({
        error: "Validation Error",
        message: "No fields to update"
      });
    }

    values.push(employeeId);

    // Update employee
    await db.execute(
      `UPDATE employees SET ${updates.join(', ')}, updated_at = NOW() WHERE employee_id = ?`,
      values
    );

    // Fetch updated employee
    const [employees] = await db.execute(
      `SELECT 
        employee_id,
        name,
        email,
        phone,
        designation,
        status,
        created_at,
        updated_at
      FROM employees 
      WHERE employee_id = ?`,
      [employeeId]
    );

    res.json({ data: employees[0] });
  } catch (error) {
    logger.error("Update employee error:", error);
    res.status(500).json({
      error: "Server Error",
      message: "Failed to update employee"
    });
  }
};

/**
 * Delete Employee
 * DELETE /api/admin/employees/:employeeId
 */
exports.deleteEmployee = async (req, res) => {
  try {
    const { employeeId } = req.params;

    if (!employeeId || isNaN(employeeId)) {
      return res.status(400).json({
        error: "Validation Error",
        message: "Invalid employee ID"
      });
    }

    const [result] = await db.execute(
      `DELETE FROM employees WHERE employee_id = ?`,
      [employeeId]
    );

    if (result.affectedRows === 0) {
      return res.status(404).json({
        error: "Employee not found",
        message: `Employee with ID ${employeeId} not found`
      });
    }

    res.json({
      success: true,
      message: "Employee deleted successfully"
    });
  } catch (error) {
    logger.error("Delete employee error:", error);
    res.status(500).json({
      error: "Server Error",
      message: "Failed to delete employee"
    });
  }
};

/**
 * Update Employee Status
 * PATCH /api/admin/employees/:employeeId/status
 */
exports.updateEmployeeStatus = async (req, res) => {
  try {
    const { employeeId } = req.params;
    const { status } = req.body;

    if (!employeeId || isNaN(employeeId)) {
      return res.status(400).json({
        error: "Validation Error",
        message: "Invalid employee ID"
      });
    }

    // Validate status
    if (!status || !['active', 'inactive'].includes(status)) {
      return res.status(400).json({
        error: "Validation Error",
        message: "Invalid status. Must be 'active' or 'inactive'"
      });
    }

    // Check if employee exists
    const [existing] = await db.execute(
      `SELECT employee_id FROM employees WHERE employee_id = ?`,
      [employeeId]
    );

    if (existing.length === 0) {
      return res.status(404).json({
        error: "Employee not found",
        message: `Employee with ID ${employeeId} not found`
      });
    }

    // Update status
    await db.execute(
      `UPDATE employees SET status = ?, updated_at = NOW() WHERE employee_id = ?`,
      [status, employeeId]
    );

    res.json({
      success: true,
      message: "Employee status updated successfully"
    });
  } catch (error) {
    logger.error("Update employee status error:", error);
    res.status(500).json({
      error: "Server Error",
      message: "Failed to update employee status"
    });
  }
};

