const db = require("../config/db");
const validateAadhaar = require("../utils/validateAadhaar");
const validatePAN = require("../utils/validatePan");
const fs = require("fs");
const path = require("path");
const emailService = require("../utils/emailService");
const logger = require("../utils/logger");

exports.registerUser = async (req, res) => {
  try {
    const {
      fullname,
      contact,
      address,
      office_address,
      office_contact,
      work_experience,
      declaration_acceptance,
      aadhaar_number,
      pan_number,
      passport_number,
      lead_type,
      employee_id,
    } = req.body;

    // VALIDATE DOCUMENT NUMBERS
    if (!validateAadhaar(aadhaar_number))
      return res.status(400).json({ error: "Invalid Aadhaar number" });

    if (!validatePAN(pan_number))
      return res.status(400).json({ error: "Invalid PAN number" });

    // Validate lead_type if provided
    if (lead_type && !['chp', 'fsp'].includes(lead_type.toLowerCase())) {
      return res.status(400).json({ error: "Invalid lead_type. Must be 'chp' or 'fsp'" });
    }

    // Validate employee_id if provided
    if (employee_id) {
      const employeeIdNum = parseInt(employee_id);
      if (isNaN(employeeIdNum) || employeeIdNum <= 0) {
        return res.status(400).json({ error: "Invalid employee_id" });
      }

      // Check if employee exists and is active
      const [employees] = await db.execute(
        `SELECT employee_id FROM employees WHERE employee_id = ? AND status = 'active'`,
        [employeeIdNum]
      );

      if (employees.length === 0) {
        return res.status(400).json({ 
          error: "Invalid employee_id or employee is inactive" 
        });
      }
    }

    const [result] = await db.execute(
      `INSERT INTO users 
            (fullname, contact, address, office_address, office_contact, work_experience, declaration_acceptance, aadhaar_number, pan_number, passport_number, lead_type, employee_id, created_at)
             VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())`,
      [
        fullname,
        contact,
        address,
        office_address,
        office_contact,
        work_experience,
        declaration_acceptance ? 1 : 0,
        aadhaar_number,
        pan_number,
        passport_number || null,
        lead_type ? lead_type.toLowerCase() : null,
        employee_id ? parseInt(employee_id) : null,
      ]
    );

    // REMOVED: Don't create folders here - they will be created when documents are uploaded
    // Folders will be created in uploadDocuments using user_name

    res.json({
      status: "success",
      user_id: result.insertId,
    });
  } catch (err) {
    logger.error("Registration error:", err);
    res.status(500).json({ 
      error: "Server Error",
      message: "Failed to register user"
    });
  }
};

exports.uploadDocuments = async (req, res) => {
  try {
    // Get user_id and user_name from req.body (sent via FormData)
    const { user_id, user_name } = req.body;

    if (!user_id) {
      return res.status(400).json({ error: "user_id is required" });
    }

    if (!user_name) {
      return res.status(400).json({ error: "user_name is required" });
    }

    // Use user_name for folder path instead of user_id
    const userPath = path.join(__dirname, "..", "uploads", "users", user_name);

    // Create user directory only when documents are uploaded (not during registration)
    if (!fs.existsSync(userPath)) {
      fs.mkdirSync(userPath, { recursive: true });
    }

    // Ensure subdirectories exist
    const subDirs = ["aadhaar", "pan", "passport", "signature", "mou"];
    subDirs.forEach((dir) => {
      const dirPath = path.join(userPath, dir);
      if (!fs.existsSync(dirPath)) {
        fs.mkdirSync(dirPath, { recursive: true });
      }
    });

    // Check if document record exists for this user
    const [existingDocs] = await db.execute(
      `SELECT * FROM user_documents WHERE user_id = ?`,
      [user_id]
    );

    const docData = {
      aadhaar_file: moveFile(
        req.files.aadhaar?.[0],
        path.join(userPath, "aadhaar")
      ),
      pan_file: moveFile(req.files.pan?.[0], path.join(userPath, "pan")),
      passport_file: moveFile(
        req.files.passport?.[0],
        path.join(userPath, "passport")
      ),
      signature_file: moveFile(
        req.files.signature?.[0],
        path.join(userPath, "signature")
      ),
      mou_pdf: moveFile(req.files.mou?.[0], path.join(userPath, "mou")),
    };

    // Build update object - only update fields that have new files
    const updateFields = [];
    const updateValues = [];
    
    if (docData.aadhaar_file) {
      updateFields.push("aadhaar_file = ?");
      updateValues.push(docData.aadhaar_file);
    }
    if (docData.pan_file) {
      updateFields.push("pan_file = ?");
      updateValues.push(docData.pan_file);
    }
    if (docData.passport_file) {
      updateFields.push("passport_file = ?");
      updateValues.push(docData.passport_file);
    }
    if (docData.signature_file) {
      updateFields.push("signature_file = ?");
      updateValues.push(docData.signature_file);
    }
    if (docData.mou_pdf) {
      updateFields.push("mou_pdf = ?");
      updateValues.push(docData.mou_pdf);
    }

    if (existingDocs.length > 0) {
      // Update existing record - merge old values with new ones
      const existingDoc = existingDocs[0];
      const mergedDocData = {
        aadhaar_file: docData.aadhaar_file || existingDoc.aadhaar_file,
        pan_file: docData.pan_file || existingDoc.pan_file,
        passport_file: docData.passport_file || existingDoc.passport_file,
        signature_file: docData.signature_file || existingDoc.signature_file,
        mou_pdf: docData.mou_pdf || existingDoc.mou_pdf,
      };

      // Only update if there are new files
      if (updateFields.length > 0) {
        updateValues.push(user_id);
        await db.execute(
          `UPDATE user_documents 
           SET ${updateFields.join(", ")} 
           WHERE user_id = ?`,
          updateValues
        );
      }
    } else {
      // Insert new record
      await db.execute(
        `INSERT INTO user_documents 
                  (user_id, aadhaar_file, pan_file, passport_file, signature_file, mou_pdf, created_at)
               VALUES (?, ?, ?, ?, ?, ?, NOW())`,
        [
          user_id,
          docData.aadhaar_file,
          docData.pan_file,
          docData.passport_file,
          docData.signature_file,
          docData.mou_pdf,
        ]
      );
    }

    // Get final document paths after update/insert
    // Use a small delay to ensure database transaction is committed
    await new Promise(resolve => setTimeout(resolve, 100));
    
    const [finalDocs] = await db.execute(
      `SELECT * FROM user_documents WHERE user_id = ?`,
      [user_id]
    );

    // Get user data for email
    const [userData] = await db.execute(
      `SELECT * FROM users WHERE id = ?`,
      [user_id]
    );

    // Only send email for NEW submissions (not updates)
    // The email will show ALL document types, with download links for uploaded ones
    if (userData.length > 0 && existingDocs.length === 0) {
      const user = userData[0];
      const finalDocData = finalDocs[0] || {};

      // Prepare document paths for email - use exact database values (don't use || null)
      const documentPaths = {
        aadhaar: finalDocData.aadhaar_file,
        pan: finalDocData.pan_file,
        passport: finalDocData.passport_file,
        signature: finalDocData.signature_file,
        mou: finalDocData.mou_pdf,
      };

      logger.debug("Document paths being sent to email service:", documentPaths);

      // Send admin notification email (only once for new submission)
      const adminEmail = process.env.ADMIN_EMAIL || "business@ftv.ind.in";
      const baseUrl = process.env.BASE_URL || "https://backend.ftvassets.in:5000";
      emailService
        .sendAdminNotification(adminEmail, { ...user, user_id }, documentPaths, baseUrl)
        .catch((err) => {
          logger.error("Failed to send admin notification email:", err);
        });
    }

    res.json({ status: "success", message: "Documents uploaded" });
  } catch (err) {
    logger.error("Upload documents error:", err);
    res.status(500).json({ 
      error: "Server Error",
      message: "Failed to upload documents"
    });
  }
};

function moveFile(file, destinationFolder) {
  if (!file) return null;

  try {
    // Ensure destination folder exists
    if (!fs.existsSync(destinationFolder)) {
      fs.mkdirSync(destinationFolder, { recursive: true });
    }

    const newPath = path.join(destinationFolder, file.filename);

    // Check if source file exists before moving
    if (!fs.existsSync(file.path)) {
      logger.error(`Source file not found: ${file.path}`);
      return null;
    }

    fs.renameSync(file.path, newPath);
    return newPath.replace(/\\/g, "/"); // return relative path
  } catch (error) {
    logger.error(`Error moving file ${file.filename}:`, error);
    return null;
  }
}
