const jwt = require("jsonwebtoken");

// JWT Secret - must be in environment variables for production
const JWT_SECRET = process.env.JWT_SECRET;

if (!JWT_SECRET || JWT_SECRET === "your-secret-key-change-in-production") {
  if (process.env.NODE_ENV === 'production') {
    throw new Error('JWT_SECRET must be set in production environment');
  }
}

// Fallback for development only
const SECRET = JWT_SECRET || "your-secret-key-change-in-production";

/**
 * Middleware to verify JWT token for admin routes
 */
const authenticateAdmin = (req, res, next) => {
  try {
    // Get token from Authorization header
    const authHeader = req.headers.authorization;
    
    if (!authHeader || !authHeader.startsWith("Bearer ")) {
      return res.status(401).json({
        error: "Unauthorized",
        message: "No token provided. Please provide a valid token in Authorization header."
      });
    }

    const token = authHeader.substring(7); // Remove "Bearer " prefix

    // Verify token
    const decoded = jwt.verify(token, SECRET);
    
    // Attach admin info to request
    req.admin = decoded;
    next();
  } catch (error) {
    if (error.name === "TokenExpiredError") {
      return res.status(401).json({
        error: "Unauthorized",
        message: "Token has expired. Please login again."
      });
    } else if (error.name === "JsonWebTokenError") {
      return res.status(401).json({
        error: "Unauthorized",
        message: "Invalid token. Please provide a valid token."
      });
    }
    
    return res.status(401).json({
      error: "Unauthorized",
      message: "Authentication failed."
    });
  }
};

module.exports = { authenticateAdmin, JWT_SECRET: SECRET };


