/**
 * Global error handler middleware
 * Should be added after all routes
 */

const logger = require("../utils/logger");

const errorHandler = (err, req, res, next) => {
  // Log error
  logger.error("Error:", {
    message: err.message,
    stack: process.env.NODE_ENV === 'production' ? undefined : err.stack,
    path: req.path,
    method: req.method,
    ip: req.ip,
    timestamp: new Date().toISOString()
  });

  // Default error
  let statusCode = err.statusCode || 500;
  let message = err.message || "Internal Server Error";

  // Database errors
  if (err.code === 'ER_DUP_ENTRY') {
    statusCode = 409;
    message = "Duplicate entry. This record already exists.";
  } else if (err.code === 'ER_NO_REFERENCED_ROW_2') {
    statusCode = 400;
    message = "Invalid reference. Referenced record does not exist.";
  } else if (err.code === 'ER_BAD_FIELD_ERROR') {
    statusCode = 500;
    message = "Database error. Please contact administrator.";
  }

  // JWT errors
  if (err.name === 'JsonWebTokenError') {
    statusCode = 401;
    message = "Invalid token.";
  } else if (err.name === 'TokenExpiredError') {
    statusCode = 401;
    message = "Token has expired.";
  }

  // Validation errors
  if (err.name === 'ValidationError') {
    statusCode = 400;
    message = err.message;
  }

  // Don't expose internal errors in production
  if (statusCode === 500 && process.env.NODE_ENV === 'production') {
    message = "Internal server error. Please try again later.";
  }

  res.status(statusCode).json({
    error: statusCode >= 500 ? "Server Error" : "Request Error",
    message: message,
    ...(process.env.NODE_ENV !== 'production' && { stack: err.stack })
  });
};

/**
 * 404 handler
 */
const notFoundHandler = (req, res) => {
  res.status(404).json({
    error: "Not Found",
    message: `Route ${req.method} ${req.path} not found`
  });
};

module.exports = {
  errorHandler,
  notFoundHandler
};

