const { body, param, query, validationResult } = require("express-validator");

/**
 * Handle validation errors
 */
const handleValidationErrors = (req, res, next) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        error: "Validation Error",
        message: "Invalid input data",
        errors: errors.array()
      });
    }
    next();
  } catch (err) {
    // If validation result itself throws an error, pass it to error handler
    next(err);
  }
};

/**
 * Registration validation rules
 */
const validateRegistration = [
  body('fullname')
    .trim()
    .notEmpty().withMessage('Full name is required')
    .isLength({ min: 2, max: 255 }).withMessage('Full name must be between 2 and 255 characters'),
  
  body('contact')
    .trim()
    .notEmpty().withMessage('Contact number is required')
    .isLength({ min: 10, max: 15 }).withMessage('Contact number must be between 10 and 15 characters'),
  
  body('address')
    .trim()
    .notEmpty().withMessage('Address is required')
    .isLength({ max: 500 }).withMessage('Address must not exceed 500 characters'),
  
  body('office_address')
    .trim()
    .notEmpty().withMessage('Office address is required')
    .isLength({ max: 500 }).withMessage('Office address must not exceed 500 characters'),
  
  body('office_contact')
    .trim()
    .notEmpty().withMessage('Office contact is required')
    .isLength({ min: 10, max: 15 }).withMessage('Office contact must be between 10 and 15 characters'),
  
  body('work_experience')
    .trim()
    .notEmpty().withMessage('Work experience is required')
    .isLength({ max: 500 }).withMessage('Work experience must not exceed 500 characters'),
  
  body('declaration_acceptance')
    .isBoolean().withMessage('Declaration acceptance must be a boolean')
    .equals(true).withMessage('Declaration acceptance must be true'),
  
  body('aadhaar_number')
    .trim()
    .notEmpty().withMessage('Aadhaar number is required')
    .isLength({ min: 12, max: 12 }).withMessage('Aadhaar number must be 12 digits')
    .isNumeric().withMessage('Aadhaar number must contain only digits'),
  
  body('pan_number')
    .trim()
    .notEmpty().withMessage('PAN number is required')
    .isLength({ min: 10, max: 10 }).withMessage('PAN number must be 10 characters')
    .matches(/^[A-Z]{5}[0-9]{4}[A-Z]{1}$/).withMessage('Invalid PAN number format'),
  
  body('passport_number')
    .optional()
    .trim()
    .isLength({ max: 20 }).withMessage('Passport number must not exceed 20 characters'),
  
  body('lead_type')
    .optional()
    .isIn(['chp', 'fsp']).withMessage('Lead type must be either chp or fsp'),
  
  body('employee_id')
    .optional()
    .isInt({ min: 1 }).withMessage('Employee ID must be a positive integer'),
  
  handleValidationErrors
];

/**
 * Admin login validation
 */
const validateLogin = [
  body('username')
    .trim()
    .notEmpty().withMessage('Username is required')
    .isLength({ min: 3, max: 50 }).withMessage('Username must be between 3 and 50 characters'),
  
  body('password')
    .trim()
    .notEmpty().withMessage('Password is required')
    .isLength({ min: 6 }).withMessage('Password must be at least 6 characters'),
  
  handleValidationErrors
];

/**
 * Employee validation
 */
const validateEmployee = [
  body('name')
    .trim()
    .notEmpty().withMessage('Name is required')
    .isLength({ min: 2, max: 255 }).withMessage('Name must be between 2 and 255 characters'),
  
  body('email')
    .optional()
    .trim()
    .isEmail().withMessage('Invalid email format')
    .normalizeEmail(),
  
  body('phone')
    .optional()
    .trim()
    .isLength({ max: 50 }).withMessage('Phone must not exceed 50 characters'),
  
  body('designation')
    .optional()
    .trim()
    .isLength({ max: 255 }).withMessage('Designation must not exceed 255 characters'),
  
  handleValidationErrors
];

/**
 * Update employee validation
 */
const validateUpdateEmployee = [
  body('name')
    .optional()
    .trim()
    .notEmpty().withMessage('Name cannot be empty')
    .isLength({ min: 2, max: 255 }).withMessage('Name must be between 2 and 255 characters'),
  
  body('email')
    .optional()
    .trim()
    .isEmail().withMessage('Invalid email format')
    .normalizeEmail(),
  
  body('phone')
    .optional()
    .trim()
    .isLength({ max: 50 }).withMessage('Phone must not exceed 50 characters'),
  
  body('designation')
    .optional()
    .trim()
    .isLength({ max: 255 }).withMessage('Designation must not exceed 255 characters'),
  
  handleValidationErrors
];

/**
 * Status update validation
 */
const validateStatus = [
  body('status')
    .notEmpty().withMessage('Status is required')
    .isIn(['active', 'inactive', 'pending']).withMessage('Status must be active, inactive, or pending'),
  
  handleValidationErrors
];

/**
 * Employee status validation
 */
const validateEmployeeStatus = [
  body('status')
    .notEmpty().withMessage('Status is required')
    .isIn(['active', 'inactive']).withMessage('Status must be active or inactive'),
  
  handleValidationErrors
];

/**
 * ID parameter validation
 */
const validateId = [
  param('userId')
    .isInt({ min: 1 }).withMessage('Invalid user ID'),
  handleValidationErrors
];

const validateEmployeeId = [
  param('employeeId')
    .isInt({ min: 1 }).withMessage('Invalid employee ID'),
  handleValidationErrors
];

module.exports = {
  validateRegistration,
  validateLogin,
  validateEmployee,
  validateUpdateEmployee,
  validateStatus,
  validateEmployeeStatus,
  validateId,
  validateEmployeeId,
  handleValidationErrors
};

