const express = require("express");
const router = express.Router();
const controller = require("../controllers/registrationController");
const multer = require("multer");
const fs = require("fs");
const path = require("path");
const { registrationLimiter } = require("../middleware/security");
const { validateRegistration } = require("../middleware/validation");
const logger = require("../utils/logger");

// Get the absolute path to uploads/tmp directory
const uploadsTmpDir = path.join(__dirname, "..", "uploads", "tmp");

// Create uploads/tmp directory if it doesn't exist
if (!fs.existsSync(uploadsTmpDir)) {
  fs.mkdirSync(uploadsTmpDir, { recursive: true });
  logger.info("Created uploads/tmp directory at:", uploadsTmpDir);
}

// File upload configuration with security
const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    // Double-check directory exists before saving
    if (!fs.existsSync(uploadsTmpDir)) {
      fs.mkdirSync(uploadsTmpDir, { recursive: true });
    }
    cb(null, uploadsTmpDir); // Use absolute path
  },
  filename: function (req, file, cb) {
    // Sanitize filename - remove special characters
    const sanitized = file.originalname.replace(/[^a-zA-Z0-9._-]/g, '_');
    cb(null, Date.now() + "_" + sanitized);
  },
});

// File filter for security
const fileFilter = (req, file, cb) => {
  // Allowed file types
  const allowedMimes = [
    'image/jpeg',
    'image/jpg',
    'image/png',
    'image/gif',
    'application/pdf'
  ];
  
  if (allowedMimes.includes(file.mimetype)) {
    cb(null, true);
  } else {
    cb(new Error('Invalid file type. Only images (JPEG, PNG, GIF) and PDF files are allowed.'), false);
  }
};

const upload = multer({ 
  storage,
  fileFilter,
  limits: {
    fileSize: 10 * 1024 * 1024, // 10MB max file size
    files: 5 // Max 5 files
  }
});

router.post("/register", registrationLimiter, validateRegistration, controller.registerUser);

router.post(
  "/upload-documents",
  registrationLimiter,
  upload.fields([
    { name: "aadhaar", maxCount: 1 },
    { name: "pan", maxCount: 1 },
    { name: "passport", maxCount: 1 },
    { name: "signature", maxCount: 1 },
    { name: "mou", maxCount: 1 },
  ]),
  (err, req, res, next) => {
    // Handle multer errors
    if (err) {
      if (err instanceof multer.MulterError) {
        if (err.code === 'LIMIT_FILE_SIZE') {
          return res.status(400).json({
            error: "File too large",
            message: "File size exceeds 10MB limit"
          });
        }
        if (err.code === 'LIMIT_FILE_COUNT') {
          return res.status(400).json({
            error: "Too many files",
            message: "Maximum 5 files allowed"
          });
        }
        return res.status(400).json({
          error: "Upload error",
          message: err.message
        });
      }
      // Handle file type validation errors
      return res.status(400).json({
        error: "Upload error",
        message: err.message || "Invalid file type or upload error"
      });
    }
    next();
  },
  controller.uploadDocuments
);

module.exports = router;
